import { Card } from "@/components/ui/Card";
import { Button } from "@/components/ui/Button";
import { Badge } from "@/components/ui/Badge";
import { supabaseServer } from "@/lib/supabase/server";

export default async function ApplicationDetail({ params }: { params: { id: string } }) {
  const supabase = supabaseServer();
  const { data: { user } } = await supabase.auth.getUser();

  const { data: app } = await supabase
    .from("applications")
    .select("id,status,amount_requested,term_requested,purpose_text,created_at,loan_product_id")
    .eq("id", params.id)
    .eq("user_id", user?.id ?? "")
    .maybeSingle();

  if (!app) {
    return (
      <Card>
        <h1 className="text-2xl font-semibold">Application not found</h1>
      </Card>
    );
  }

  const { data: routes } = await supabase
    .from("application_routing")
    .select("id,status,issuer_id,sent_at,last_action_at")
    .eq("application_id", app.id)
    .order("sent_at", { ascending: false });

  async function submitAction() {
    "use server";
  }

  return (
    <div className="space-y-6">
      <Card>
        <div className="flex items-start justify-between">
          <div>
            <h1 className="text-2xl font-semibold">Application #{String(app.id).slice(0, 8)}</h1>
            <p className="mt-1 text-sm text-slate-600">Amount P{app.amount_requested} • Term {app.term_requested} months</p>
            {app.purpose_text ? <p className="mt-2 text-sm text-slate-600">Purpose: {app.purpose_text}</p> : null}
          </div>
          <Badge tone="blue">{app.status}</Badge>
        </div>

        <div className="mt-5 flex gap-3">
          <form action="/api/applications/submit" method="POST">
            <input type="hidden" name="application_id" value={app.id} />
            <Button type="submit" disabled={app.status !== "draft"}>
              {app.status === "draft" ? "Submit & Route to Issuers" : "Submitted"}
            </Button>
          </form>
          <form action="/api/applications/delete" method="POST">
            <input type="hidden" name="application_id" value={app.id} />
            <Button variant="secondary" type="submit">Delete</Button>
          </form>
        </div>

        <p className="mt-3 text-xs text-slate-500">
          Submitting will route your application to multiple approved issuers based on eligibility rules.
        </p>
      </Card>

      <Card>
        <div className="text-sm font-semibold">Routing</div>
        <div className="mt-3 space-y-2">
          {(routes ?? []).length === 0 ? (
            <div className="text-sm text-slate-600">Not routed yet.</div>
          ) : (
            (routes ?? []).map((r) => (
              <div key={r.id} className="rounded-xl2 border border-thebe-border bg-slate-50 px-4 py-3 flex items-center justify-between">
                <div className="text-sm text-slate-700">
                  Issuer: {String(r.issuer_id).slice(0, 8)} • Sent: {r.sent_at ? new Date(r.sent_at).toLocaleString() : "-"}
                </div>
                <Badge tone="blue">{r.status}</Badge>
              </div>
            ))
          )}
        </div>
      </Card>
    </div>
  );
}
